/**
 * PenguinAttackBlock.java
 *
 * Description: This ADT represents a block on a Penguin Attack board.  It was
 * designed for use with the PAAI class, but should be complete enough for any 
 * representation of a Penguin Attack board state.
 *
 * Copyright (c) 2011, Jason Buck
 * 
 * Distributed under the BSD-new license. For details see the BSD_LICENSE file 
 * that should have been included with this distribution. If the source you 
 * acquired this distribution from incorrectly removed this file, the license 
 * may be viewed at http://www.opensource.org/licenses/bsd-license.php.
 */

public class PenguinAttackBlock extends PenguinAttackAIClient
{
	public static final String POSSIBLE_STATE_VALUES = ".f<>xcbwh*q?";
	
	private int chainValue;
	@SuppressWarnings("unused")
	private int garbageDepth;
	private char type;
	private int color;
	private char state;
	
	/**
	 * Constructs an object of ADT PenguinAttackBlock
	 *
	 * @param attributeString	A 4-character string representing a PA  block.
	 * each char from left to right represents a block attribute.
	 *
	 * attributeString.charAt(0):  The chain value of the block if it is a
	 * normal block, the garbage depth of the block if it is a garbage block.
	 * Value must be a char representation of a  non-negative number. The chain
	 * value of a block is probably something the AI will consider, but I the
	 * garbage depth probably isn't.
	 *
	 *
	 * attributeString.charAt(1): The type of the block.  Currently the type can
	 * be 'n' for normal, 'g' for garbage, or '_' for and empty space. There is 
	 * one more type in TA which we have ignored so far which are the special
	 * "stone" bonus blocks given for gaining the opportunity to intentionally 
	 * raise the stack.
	 *
	 *
	 * attributeString.charAt(2): The color of the block given as an integer
	 * representation.
	 * 
	 * For normal blocks:
	 * '1' for purple
	 * '2' for red
	 * '3' for green
	 * '4' for yellow
	 * '5' for light blue
	 * '6' for dark blue
	 *
	 * For garbage blocks:
	 * Valid values are 0-9, though this information is unimportant to an AI. We
	 * are tracking it for completeness of state representation in case we do
	 * something crazy like make the AI maintain its own state.  It's worth
	 * noting that '0' and '1' may have two different meanings; to determine
	 * which meaning is intended, you must look at garbageDepth.
	 *
	 *
	 * attributeString.charAt(3): The current state of the block.  You can check
	 * to see if a char reresents a valid state by testing that it is contained 
	 * in the string PenguinAttackBlock.POSSIBLE_STATE_VALUES.
	 *
	 * '.' - Block is idle
	 * 'f' - Block is falling
	 * '<' - Block is swapping to the left
	 * '>' - Block is swapping to the right
	 * 'x' - Block is in an unbroken combo
	 * 'c' - Block is in an unbroken combo that is a part of a chain
	 * 'b' - Block is busy (not sure what this means exactly)
	 * 'w' - Block is waiting (not sure what this means exactly)
	 * 'h' - Block is hanging
	 * '*' - Block is breaking
	 * 'q' - Block is on the queued line which is currently unavailable to the
     *       player
	 * '?' - Block is in an unknown state (this should never happen).
	 */
	public PenguinAttackBlock(String attributeString) throws Exception
	{
		if (!this.setType(attributeString.charAt(1)))
			throw new Exception("Invalid type parameter '" + attributeString.charAt(1) +"' given to PenguinAttackBlock constructor.");
				
		if (this.type == '_')
			this.color = 0;
		else
		{
			if (!this.setColor(Integer.valueOf(attributeString.charAt(2)) - 48))
				throw new Exception("Invalid color parameter '" + attributeString.charAt(2) +"' given to PenguinAttackBlock constructor.");
		}
		
		if (this.type == '_')
			this.state = '.';
		else
		{
			if (!this.setState(attributeString.charAt(3)))
			throw new Exception("Invalid state parameter '" + attributeString.charAt(3) +"' given to PenguinAttackBlock constructor.");
		}
			
		if (this.type == 'n' && !this.setChainValue(Integer.valueOf(attributeString.charAt(0)) - 48))
			throw new Exception("Invalid chain value parameter '" + attributeString.charAt(0) +"' given to PenguinAttackBlock constructor");
		
		if (this.type == 'g' && !this.setGarbageDepth(Integer.valueOf(attributeString.charAt(0)) - 48))
			throw new Exception("Invalid garbage depth parameter '" + attributeString.charAt(0) +"' given to PenguinAttackBlock constructor");
	}
	
	/**
	 * Chain value accessor (PAAI doesn't currently use this, but likely will as
	 * it learns how to make chains).
	 *
	 * @return int 
	 *
	 * The chain value of the block.
	 */
	public int getChainValue()
	{
		return this.chainValue;
	}
	
	/**
	 * Chain value mutator - this is only relevant to normal blocks.  This is
	 * enforced in code.  If a garbage block is converted to a normal block, its
	 * type must be changed before its value may be. (With the way I have 
	 * implemented PAAI, I don't expect this to ever get used, which is why it
	 * is private.  The constructor uses it).
	 *
	 * @param chainValue
	 *
	 * Value must be non-negative.
	 *
	 * @return boolean 
	 *
	 * Returns true if the parameter chainValue was valid and the chainValue was
	 * set.
	 */
	private boolean setChainValue(int chainValue)
	{
		if (this.type == 'n' && chainValue >= 0)
		{
			this.chainValue = chainValue;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Garbage depth mutator - only relevant to garbage blocks.  This is
	 * enforced by this code. (PAAI doesn't use this, hence it is private. It is
	 * used by the constructor).
	 *
	 * @param garbageDepth
	 *
	 * Valid values are non-negative.
	 *
	 * @return boolean 
	 *
	 * Returns true if the parameter state was valid and the state was set.
	 */
	private boolean setGarbageDepth(int garbageDepth)
	{
		if (this.type == 'g')
		{
			this.garbageDepth = garbageDepth;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Checks to see if a block is in a normal block.
	 * 
	 * @return boolean true if block is normal.
	 */
	public boolean isNormalBlock()
	{
		if (this.type == 'n')
			return true;
		return false;
	}

	/**
	 * Checks to see if a block is in a garbage block.
	 * 
	 * @return boolean true if block is garbage.
	 */
	public boolean isGarbageBlock()
	{
		if (this.type == 'g')
			return true;
		return false;
	}
	
	/**
	 * Checks to see if a block is actually an empty space.
	 * 
	 * @return boolean true if block is actually an empty space.
	 */
	public boolean isEmptySpace()
	{
		if (this.type == '_')
			return true;
		return false;
	}
	
	/**
	 * Type mutator - This should only be used when we change a garbage block
	 * into a normal block.  This is not enforced by this code. (Not used by
	 * PAAI, used by constructor, hence private).
	 *
	 * @param type
	 *
	 * State must be a char found in the string 
	 * PenguinAttackBlock.POSSIBLE_STATE_VALUES (see constructor documentation).
	 *
	 * @return boolean 
	 *
	 * Returns true if the parameter type was valid and the type was set.
	 */
	private boolean setType(char type)
	{
		if (type == 'n' || type == 'g' || type == '_')
		{	
			this.type = type;
			return true;
		}
			
		return false;
	}
	
	/**
	 * Color accessor (I thought about encapsulating the color values, such that
	 * this was either several methods, e.g. isDarkBlue, or such that it
	 * returned a string name of the color.  In practice the way this is used
	 * is 
	 * 
	 * if (board[i][j].getColor() == board[i][k].getColor())
	 *
	 * which seems clear enough.
	 *
	 * @return int 
	 *
	 * The color value of the block (see constructor documentation).
	 */
	public int getColor()
	{
		return this.color;
	}
	
	/**
	 * Color mutator - This should only be used if we convert a garbage block to
	 * a normal block, or if a garbage block becomes a different part of a
	 * smaller garbage block.  Normal blocks never change color. This is not
	 * enforced by this code. (Not used by PAAI, used by constructor, hence 
	 * private).
	 *
	 * @param color
	 *
	 * Valid values are 0-9 (see constructor documentation).
	 *
	 * @return boolean 
	 *
	 * Returns true if the parameter color was valid and the color was set.
	 */
	private boolean setColor(int color)
	{
		if (color >=0 && color <=9)
		{
			this.color = color;
			return true;
		}
		
		return false;
	}
	
	/**
	 * Checks to see if a block is in an idle state.
	 * 
	 * @return boolean true if block is idle.
	 */
	public boolean blockIsIdle()
	{
		if (state == '.')
			return true;
		return false;
	}
		
	/**
	 * State mutator.  (Used by the constructor, not PAAI.  Hence Private.)
	 *
	 * @param state
	 *
	 * State must be a char found in the string 
	 * PenguinAttackBlock.POSSIBLE_STATE_VALUES (see constructor documentation).
	 *
	 * @return boolean 
	 *
	 * Returns true if the parameter state was valid and the state was set.
	 */
	private boolean setState(char state)
	{
		if (POSSIBLE_STATE_VALUES.contains(String.valueOf(state))) {
			this.state = state;
			return true;
		}
		
		return false;
	}
}